<?php
/*--------------------------------------------------------------
   GambioConversionRateCount.php 2021-08-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition;

use Doctrine\DBAL\Connection;
use Gambio\Admin\Modules\Statistics\App\Data\Factory as DataFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory as OverviewFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Option\Predefined\TimespanDropdown;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Options;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data;
use NumberFormatter;

class GambioConversionRateCount extends GambioWidgetDefinition
{
    private const TYPE = 'gambio.conversionrate.count';
    
    private const VERSION = '0.1.0';
    
    private const ICON = "data:image/png;base64,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";
    
    private const WIDGET_TITLE = [
        self::LANGUAGE_CODE_GERMAN  => 'Konversionsrate',
        self::LANGUAGE_CODE_ENGLISH => 'Conversion rate'
    ];
    
    /**
     * @var DataFactory
     */
    private $dataFactory;
    
    /**
     * @var Connection
     */
    private $connection;
    
    /**
     * @var NumberFormatter
     */
    private $numberFormatter;
    
    
    /**
     * @inheritDoc
     */
    public function __construct(
        OverviewFactory $overviewFactory,
        DataFactory $dataFactory,
        Connection $connection,
        NumberFormatter $numberFormatter
    ) {
        $this->dataFactory     = $dataFactory;
        $this->connection      = $connection;
        $this->numberFormatter = $numberFormatter;
        
        parent::__construct($overviewFactory->createType(self::TYPE),
                            $overviewFactory->createVersion(self::VERSION),
                            $overviewFactory->createIconUsingData(self::ICON),
                            $overviewFactory->useVisualizations()->createText(),
                            $overviewFactory->useOptions()->createOptions($overviewFactory->useOptions()
                                                                              ->usePredefined()
                                                                              ->createTimespanDropdownIncludingToday($overviewFactory)),
                            $overviewFactory->createTitles($overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_GERMAN),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_GERMAN]),
                                                           $overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_ENGLISH),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_ENGLISH])));
    }
    
    
    /**
     * @inheritDoc
     */
    public function data(Options $options): Data
    {
        $conversionValues = [];
        $timespan         = $this->dataFactory->useTimespan()->createFromTerm($options->getById(TimespanDropdown::ID)
                                                                                  ->value());
        $parameters       = [
            ':excludedOrderIds' => implode(',', self::EXCLUDED_ORDER_STATUS_IDS),
            ':startDate'        => $timespan->startDate()->format(self::QUERY_TIMESPAN_FORMAT_DATE_START),
            ':endDate'          => $timespan->endDate()->format(self::QUERY_TIMESPAN_FORMAT_DATE_END)
        ];
        
        $orders   = $this->connection->createQueryBuilder()
            ->select('UNIX_TIMESTAMP(DATE(date_purchased)) AS date, COUNT(*) AS orders')
            ->from('orders')
            ->where('orders_status NOT IN (:excludedOrderIds)')
            ->andWhere('date_purchased BETWEEN :startDate AND :endDate')
            ->groupBy('date')
            ->orderBy('date')
            ->setParameters($parameters)
            ->execute()
            ->fetchAll();
        $visitors = $this->connection->createQueryBuilder()
            ->select('UNIX_TIMESTAMP(DATE(gm_counter_date)) AS date, gm_counter_visits_total AS visitors')
            ->from('gm_counter_visits')
            ->where('gm_counter_date BETWEEN :startDate AND :endDate')
            ->orderBy('date')
            ->setParameters($parameters)
            ->execute()
            ->fetchAll();
        
        foreach ($visitors as $visitor) {
            $visitorDate  = $visitor['date'];
            $visitorCount = (float)$visitor['visitors'];
            
            $hasOrders = false;
            foreach ($orders as $order) {
                $orderDate  = $order['date'];
                $orderCount = (float)$order['orders'];
                
                if ($visitorDate !== $orderDate) {
                    continue;
                }
                
                $hasOrders          = true;
                $conversionValues[] = $orderCount / $visitorCount;
            }
            
            if (!$hasOrders) {
                $conversionValues[] = 0.0;
            }
        }
        
        $conversionRate = array_sum($conversionValues) / sizeof($conversionValues);
        
        return $this->dataFactory->useTextData()->createTextData($this->dataFactory->useTextData()
                                                                     ->createValue($this->numberFormatter->format(is_nan($conversionRate) ? 0 : $conversionRate)));
    }
}